/*:
 * @target MZ
 * @plugindesc 物理攻撃時に確率で相手DEFを0扱い＋発動時SE再生＋習得スキルの<IgnoreDef>をパッシブ加算
 * @author ChatGPT
 * @help
 * 【使い方】
 * スキル/武器/防具/（パッシブ）習得スキルのメモ欄に以下で確率を指定：
 *   <IgnoreDef: 30%>  または  <IgnoreDef: 0.3>
 *
 * ★パッシブ機能
 *   アクターが「習得しているスキル」のメモ欄にある <IgnoreDef> を、
 *   すべてパッシブとして加算します（物理行動時に常時有効）。
 *   ※現在使用しているスキルに書かれた値は二重加算防止のためパッシブ合算から除外。
 *
 * ★効果音（任意）
 *   <IgnoreDefSE: Name, vol, pitch, pan>  // 1行指定
 *   または個別指定：
 *   <IgnoreDefSEName: Name>
 *   <IgnoreDefSEVol: 90>
 *   <IgnoreDefSEPitch: 100>
 *   <IgnoreDefSEPan: 0>
 *   優先度：スキル指定 ＞ 装備指定 ＞ プラグイン既定値
 *
 * 適用範囲：命中タイプが「物理」の行動のみ。発動時にのみSEを再生します。
 *
 * 互換性：ダメージ式評価をフックする他プラグインとは読み込み順で調整してください。
 *
 * @param DefaultSEName
 * @text 既定SE：名前
 * @type file
 * @dir audio/se/
 * @default
 *
 * @param DefaultSEVol
 * @text 既定SE：音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 *
 * @param DefaultSEPitch
 * @text 既定SE：ピッチ
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @param DefaultSEPan
 * @text 既定SE：位相
 * @type number
 * @min -100
 * @max 100
 * @default 0
 */

(() => {
  "use strict";

  const PLUGIN_NAME = document.currentScript
    ? document.currentScript.src.split("/").pop().replace(/\.js$/, "")
    : "IgnoreDefByRateMZ";
  const params = PluginManager.parameters(PLUGIN_NAME);

  const DEFAULT_SE = {
    name: String(params["DefaultSEName"] || ""),
    volume: Number(params["DefaultSEVol"] || 90),
    pitch: Number(params["DefaultSEPitch"] || 100),
    pan: Number(params["DefaultSEPan"] || 0),
  };

  const META_KEY_RATE = "IgnoreDef";
  const META_KEY_SE = "IgnoreDefSE";
  const meta = (o) => (o && o.meta) || {};

  // ---------- utils ----------
  function clamp(v, min, max) {
    return Math.max(min, Math.min(max, v));
  }

  function readIgnoreDefRate(obj) {
    if (!obj) return 0;
    let raw = meta(obj)[META_KEY_RATE] ?? null;
    if (raw == null && typeof obj.note === "string") {
      const m = obj.note.match(/<\s*IgnoreDef\s*:\s*([0-9.]+)\s*%?\s*>/i);
      if (m) raw = m[1];
    }
    if (raw == null) return 0;
    let s = String(raw).trim().replace(/\s*%$/, "");
    const n = Number(s);
    if (!isFinite(n)) return 0;
    const v = n > 1 ? n / 100 : n;
    return clamp(v, 0, 1);
  }

  function parseSeCsv(s) {
    const parts = s.split(",").map(t => t.trim()).filter(t => t.length);
    const out = {};
    if (parts[0]) out.name = parts[0];
    if (parts[1] != null) out.volume = Number(parts[1]);
    if (parts[2] != null) out.pitch  = Number(parts[2]);
    if (parts[3] != null) out.pan    = Number(parts[3]);
    return out;
  }

  function readSeSpec(obj) {
    if (!obj) return null;
    let line = meta(obj)[META_KEY_SE];
    if (line == null && typeof obj.note === "string") {
      const m = obj.note.match(/<\s*IgnoreDefSE\s*:\s*([^>]+)\s*>/i);
      if (m) line = m[1].trim();
    }
    let spec = null;
    if (line) spec = parseSeCsv(String(line));
    const name  = meta(obj)["IgnoreDefSEName"];
    const vol   = meta(obj)["IgnoreDefSEVol"];
    const pitch = meta(obj)["IgnoreDefSEPitch"];
    const pan   = meta(obj)["IgnoreDefSEPan"];
    if (name || vol || pitch || pan) {
      spec ||= {};
      if (name)  spec.name   = String(name);
      if (vol!=null)   spec.volume = Number(String(vol).replace(/\D/g, ""));
      if (pitch!=null) spec.pitch  = Number(String(pitch).replace(/\D/g, ""));
      if (pan!=null)   spec.pan    = Number(String(pan).replace(/[^\-0-9]/g, ""));
    }
    if (!spec) return null;
    return {
      name:   spec.name ?? DEFAULT_SE.name,
      volume: isFinite(spec.volume) ? spec.volume : DEFAULT_SE.volume,
      pitch:  isFinite(spec.pitch)  ? spec.pitch  : DEFAULT_SE.pitch,
      pan:    isFinite(spec.pan)    ? spec.pan    : DEFAULT_SE.pan,
    };
  }

  // パッシブ：習得スキルから合算（使用中itemは除外）
  function passiveRateFromLearnedSkills(user, currentItem) {
    if (!user) return 0;
    // アクターのみ（敵は対象外）
    if (!(user instanceof Game_Actor)) return 0;
    const skills = user.skills ? user.skills() : [];
    let sum = 0;
    for (const s of skills) {
      if (!s) continue;
      if (currentItem && s === currentItem) continue; // 二重加算防止
      sum += readIgnoreDefRate(s);
    }
    return sum;
  }

  function calcTotalRate(user, item) {
    let rate = 0;

    // 使用スキル（その行動）
    rate += readIgnoreDefRate(item);

    // 装備（武器/防具）
    if (user && typeof user.equips === "function") {
      for (const eq of user.equips()) {
        rate += readIgnoreDefRate(eq);
      }
    }

    // パッシブ（習得スキル）
    rate += passiveRateFromLearnedSkills(user, item);

    return clamp(rate, 0, 1);
  }

  function pickSe(user, item) {
    // 優先：スキル → 装備（最初に見つかったもの）→ 既定値（名前空は無音）
    const fromItem = readSeSpec(item);
    if (fromItem && fromItem.name) return fromItem;

    if (user && typeof user.equips === "function") {
      for (const eq of user.equips()) {
        const se = readSeSpec(eq);
        if (se && se.name) return se;
      }
    }
    return DEFAULT_SE.name ? DEFAULT_SE : null;
  }

  // ---------- DEF一時0化 ----------
  const _Game_BattlerBase_param = Game_BattlerBase.prototype.param;
  Game_BattlerBase.prototype.param = function (paramId) {
    const v = _Game_BattlerBase_param.call(this, paramId);
    if (this._tmpIgnoreDefFlag && paramId === 3) return 0; // 3=DEF
    return v;
  };

  // ---------- 発動判定＋SE ----------
  const _Game_Action_evalDamageFormula = Game_Action.prototype.evalDamageFormula;
  Game_Action.prototype.evalDamageFormula = function (target) {
    const item = this.item();
    const user = this.subject();

    const isPhysical = this.isPhysical();
    const totalRate = isPhysical ? calcTotalRate(user, item) : 0;
    const trigger = totalRate > 0 && Math.random() < totalRate;

    if (trigger) {
      target._tmpIgnoreDefFlag = true;
      const se = pickSe(user, item);
      if (se && se.name) AudioManager.playSe(se);
    }

    try {
      return _Game_Action_evalDamageFormula.call(this, target);
    } finally {
      if (trigger) target._tmpIgnoreDefFlag = false;
    }
  };
})();
